/* $Id: e1562tput.c,v 1.11 1998/07/20 22:49:18 ericb Exp $ */
/* Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Bryan Murray */

/* This file provides example routines to help perform a throughput to
 * an E1562 with a LIF file system installed.  All exported routines have
 * descriptions in the file e1562tput.h which must be included in your
 * C source file in order to compile.  This file should be compiled and
 * linked into the final executable.  The routines provided in this file
 * should be general purpose enough to work with an E1430, E1431, E1432,
 * or E1433, however, it has been tested with only the E1432.  The
 * code used with the E1432 can be found in the files tput.c and pbck.c.
 *
 * The routines used to provide post-processing capability make use of
 * an E1562 sequence.  This method provides a continuous stream of bytes
 * to the post-process code where the E1562 usually has the bytes already
 * in memory when the post-process code needs them.  This method is somewhat
 * faster than the file I/O method, but requires that all data which was
 * recorded to the LIF volume, be read from the volume unless additional
 * E1562 sequence operations are added which throw away data which is not
 * needed.  This requirement to read all data is not a problem if a full
 * scan of data does not exceed the space available in the E1562 shared RAM.
 * If a scan exceeds the shared RAM size, try one of the following:  1) use
 * some other RAM for playback probably in A24 space in the VXI cardcage.
 * 2) tell the E1562 that the scan size is 1/2 what it actually is and
 * process the data in 2 phases.  3) add Seq_bitBucket operations into the
 * E1562 sequence to discard unwanted data.
 *
 * The other method of doing playback is to use the LIF file system to seek
 * and read each channel's bytes individually.  This method makes overlap
 * processing easier, but is somewhat slower than the sequence method.
 */

#include "e1562tput.h"
#include <e1562lif.h>
#include <sicl.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <limits.h>
#include <math.h>
#include <stdio.h>



/* Define several constants which are used throughout the file.
 */
#define CMD_BUFFER_SIZE                 256
#define DEFAULT_TUNIT_BLOCKS            128
#define DEFAULT_SESSION_NUMBER          4
#define DEFAULT_SEQUENCE_NUMBER         4




/* Enumerate the list of all E1562 sequence operations.  Only a few of these
 * operations are actually used in this file, but the list may be useful
 * in other programs which interact with the E1562.
 */
typedef enum
{
    Seq_doNothing                       = 0x0000,
    Seq_stop                            = 0x0001,
    Seq_pauseNmsec                      = 0x0002,
    Seq_ttltrgControl                   = 0x0003,
    Seq_newSequence                     = 0x0004,
    Seq_newSequenceIfCount              = 0x0005,
    Seq_ttltrgArm                       = 0x0006,
    Seq_ttltrgWait                      = 0x0007,
    Seq_IRQarm                          = 0x0008,
    Seq_IRQwait                         = 0x0009,
    Seq_pauseNloops                     = 0x000a,

    /* Local bus throughput operations */
    Seq_lbusConsume                     = 0x1000,
    Seq_lbusEavesdrop                   = 0x1001,
    Seq_lbusConsumePipe                 = 0x1002,
    Seq_lbusEavesPipe                   = 0x1003,

    Seq_lbusConsumeFast                 = 0x1100,
    Seq_lbusEavesdropFast               = 0x1101,
    Seq_lbusConsumePipeFast             = 0x1102,
    Seq_lbusEavesPipeFast               = 0x1103,

    /* Local bus playback operations */
    Seq_lbusGenerate                    = 0x2000,
    Seq_lbusAppend                      = 0x2001,

    /* VME bus throughput operations */
    Seq_readA16Buff16                   = 0x3000,
    Seq_readA16Buff32                   = 0x3001,
    Seq_readA16BuffD32                  = 0x3002,
    Seq_readA24Buff16                   = 0x3003,
    Seq_readA24Buff32                   = 0x3004,
    Seq_readA24BuffD32                  = 0x3005,
    Seq_readA32Buff16                   = 0x3006,
    Seq_readA32Buff32                   = 0x3007,
    Seq_readA32BuffD32                  = 0x3008,
    Seq_readA16Fifo16                   = 0x3009,
    Seq_readA16Fifo32                   = 0x300a,
    Seq_readA16FifoD32                  = 0x300b,
    Seq_readA24Fifo16                   = 0x300c,
    Seq_readA24Fifo32                   = 0x300d,
    Seq_readA24FifoD32                  = 0x300e,
    Seq_readA32Fifo16                   = 0x300f,
    Seq_readA32Fifo32                   = 0x3010,
    Seq_readA32FifoD32                  = 0x3011,
    Seq_readSharedRam                   = 0x3012,
    Seq_dummyBytes                      = 0x3100,

    Seq_procReadA16Buff16               = 0x3200,
    Seq_procReadA24Buff16               = 0x3203,
    Seq_procReadA32Buff16               = 0x3206,
    Seq_procReadSharedRam               = 0x3212,

    /* VME throughput with monitor */
    Seq_readSramMonitorSram             = 0x3812,
    Seq_readSramMonitorA24buffD32       = 0x3912,
    Seq_readSramMonitorA24buff          = 0x3a12,
    Seq_readA16FifoD32MonitorSram       = 0x380b,
    Seq_readA16FifoD32MonitorA24buffD32 = 0x390b,
    Seq_readA16FifoD32MonitorA24buff    = 0x3a0b,
    Seq_readA16Fifo16MonitorSram        = 0x3809,
    Seq_readA16Fifo16MonitorA24buffD32  = 0x3909,
    Seq_readA16Fifo16MonitorA24buff     = 0x3a09,
    Seq_readA16Buff16MonitorSram        = 0x3800,
    Seq_readA16Buff16MonitorA24buffD32  = 0x3900,
    Seq_readA16Buff16MonitorA24buff     = 0x3a00,
    Seq_readA16BuffD32MonitorSram       = 0x3802,
    Seq_readA16BuffD32MonitorA24buffD32 = 0x3902,
    Seq_readA16BuffD32MonitorA24buff    = 0x3a02,

    Seq_readA24FifoD32MonitorSram       = 0x380e,
    Seq_readA24FifoD32MonitorA24buffD32 = 0x390e,
    Seq_readA24FifoD32MonitorA24buff    = 0x3a0e,
    Seq_readA24Fifo16MonitorSram        = 0x380c,
    Seq_readA24Fifo16MonitorA24buffD32  = 0x390c,
    Seq_readA24Fifo16MonitorA24buff     = 0x3a0c,
    Seq_readA24Buff16MonitorSram        = 0x3803,
    Seq_readA24Buff16MonitorA24buffD32  = 0x3903,
    Seq_readA24Buff16MonitorA24buff     = 0x3a03,
    Seq_readA24BuffD32MonitorSram       = 0x3805,
    Seq_readA24BuffD32MonitorA24buffD32 = 0x3905,
    Seq_readA24BuffD32MonitorA24buff    = 0x3a05,

    /* VME bus playback operations */
    Seq_writeA16Buff16                  = 0x4000,
    Seq_writeA16Buff32                  = 0x4001,
    Seq_writeA16BuffD32                 = 0x4002,
    Seq_writeA24Buff16                  = 0x4003,
    Seq_writeA24Buff32                  = 0x4004,
    Seq_writeA24BuffD32                 = 0x4005,
    Seq_writeA32Buff16                  = 0x4006,
    Seq_writeA32Buff32                  = 0x4007,
    Seq_writeA32BuffD32                 = 0x4008,
    Seq_writeA16Fifo16                  = 0x4009,
    Seq_writeA16Fifo32                  = 0x400a,
    Seq_writeA16FifoD32                 = 0x400b,
    Seq_writeA24Fifo16                  = 0x400c,
    Seq_writeA24Fifo32                  = 0x400d,
    Seq_writeA24FifoD32                 = 0x400e,
    Seq_writeA32Fifo16                  = 0x400f,
    Seq_writeA32Fifo32                  = 0x4010,
    Seq_writeA32FifoD32                 = 0x4011,
    Seq_writeSharedRam                  = 0x4012,
    Seq_bitBucket                       = 0x4100,

    /* Local bus throughput operations with monitor */
    Seq_lbusConsumeMonitorSharedRam     = 0x5000,
    Seq_lbusEavesdropMonitorSharedRam   = 0x5001,
    Seq_lbusConsumePipeMonitorSharedRam = 0x5002,
    Seq_lbusEavesPipeMonitorSharedRam   = 0x5003,

    Seq_lbusConsumeMonitorA24Buff16     = 0x5014,
    Seq_lbusEavesdropMonitorA24Buff16   = 0x5015,
    Seq_lbusConsumePipeMonitorA24Buff16 = 0x5016,
    Seq_lbusEavesPipeMonitorA24Buff16   = 0x5017,

    Seq_waitBitSetA16                   = 0x6000,
    Seq_waitBitClearA16                 = 0x6001,
    Seq_waitBitSetA24                   = 0x6002,
    Seq_waitBitClearA24                 = 0x6003,
    Seq_waitBitSetA32                   = 0x6004,
    Seq_waitBitClearA32                 = 0x6005,
    Seq_waitBitSetSharedRam             = 0x6006,
    Seq_waitBitClearSharedRam           = 0x6007,
    Seq_waitA16Count16                  = 0x6008,
    Seq_waitA24Count16                  = 0x6009,
    Seq_waitA32Count16                  = 0x600a,
    Seq_waitCountSharedRam16            = 0x600b,
    Seq_waitA16Count32                  = 0x600c,
    Seq_waitA24Count32                  = 0x600d,
    Seq_waitA32Count32                  = 0x600e,
    Seq_waitCountSharedRam32            = 0x600f,
    Seq_waitFifoEmpty                   = 0x6010,
    Seq_waitFifoHalfEmpty               = 0x6011,

    Seq_controlA16Reg16                 = 0x6018,
    Seq_controlA24Reg16                 = 0x6019,
    Seq_controlA32Reg16                 = 0x601a,
    Seq_controlRegSharedRam16           = 0x601b,
    Seq_controlA16Reg32                 = 0x601c,
    Seq_controlA24Reg32                 = 0x601d,
    Seq_controlA32Reg32                 = 0x601e,
    Seq_controlRegSharedRam32           = 0x601f,

    Seq_readA16Reg32                    = 0x611c,

    Seq_dumpA24SeqBytes                 = 0x6020,
    Seq_dumpA32SeqBytes                 = 0x6021,
    Seq_dumpSharedRamSeqBytes           = 0x6022,

    /* Verification operations */
    Seq_writeLocal                      = 0xf000, /* non-DMA playback */
    Seq_readLocal                       = 0xf001  /* non-DMA throughput */
} E1562SequenceOperation;


/* Define a structure used to pass information between functions.  An address
 * of this structure is passed to the calling function to be used as a
 * handle to the E1562.  This structure is opaque to calling routines -- its
 * contents are known only to this file.
 */
typedef struct
{
    double totalBytes;
    unsigned long scsiBlock;
    unsigned long scsiBlocksize;
    e1562_FILE* file;
    short* sram;
    INST id;
    e1562ID lifid;
    short session;
    short splitVolume;
    short sequenceIsThroughput;
    short debugLevel;
    short logicalAddress;
    char* interface;
} ThroughputInformation;


/* Enumerate the E1562 SCSI buses.
 */
typedef enum
{
    eScsiA,
    eScsiB
} eE1562Bus;


typedef enum
{
    SBT_UNKNOWN,
    SBT_A_ONLY,
    SBT_B_ONLY,
    SBT_A_AND_B
} ScsiBusType;



#define MAX_OPEN_FILES 20
static ThroughputInformation* openFiles[MAX_OPEN_FILES];
static void (*privateCallbackFunc)(E1562Handle, volatile void*, double) = 0;
static volatile void* privateCallbackArg;
static E1562Handle privateHandle;
static short privateRecordFinished;
static double privateRecordBytes;




/* Send a command to an E1562 module.  A command is specified with a format
 * string and parameters, just as in printf.  Thus the following invocations
 * of this function are both valid:
 *     e1562_write(e1562, 1, "mmem:scsi5:open A,0,0,#h40");
 *     e1562_write(e1562, 1, "mmem:scsi%d:open %c,%d,0,#h40",
 *                 logicalDevice, controller == 0 ? 'A' : 'B', busAddress);
 *
 * If a SICL error is returned it is reported on stderr.
 *
 * When the flag echo is non-zero, the resulting command is printed to stdout
 * before it is sent to the E1562.
 */
static long e1562_write(INST e1562, unsigned char echo, const char* cmd, ...)
{
    long error;
    va_list args;
    static char cmdbuf[CMD_BUFFER_SIZE];
    unsigned long actualcnt;

    va_start(args, cmd);
    (void) vsprintf(cmdbuf, cmd, args);
    va_end(args);

    if (cmdbuf[strlen(cmdbuf) - 1] != '\n')
    {
        (void) strcat(cmdbuf, "\n");
    }

    if (echo != 0)
    {
        printf("E1562 write: %s", cmdbuf);
    }

    error = iwrite(e1562, cmdbuf, strlen(cmdbuf), 1, &actualcnt);
    if (error != I_ERR_NOERROR)
    {
        fprintf(stderr, "E1562 write: %s -- SICL ERROR %d\n", cmdbuf, error);
    }

    return error;
}


/* Read a response from an E1562 module.  This functions assumes that a
 * command has already been sent to the module.
 *
 * If a SICL error is returned it is reported on stderr.
 *
 * When the flag echo is non-zero, the response is printed to stdout after
 * it is read from the E1562.
 */
static long e1562_read(INST e1562, unsigned char echo,
                       char * answer, unsigned long answerSize)
{
    long error;
    unsigned long actualcnt;
    int reason;

    error = iread(e1562, answer, answerSize, &reason, &actualcnt);
    if (error != I_ERR_NOERROR)
    {
        fprintf(stderr, "E1562 read:  SICL ERROR %d\n", error);
    }
    else if (echo != 0)
    {
        answer[actualcnt] = '\0';
        if (echo != 0) printf("E1562 read:  %s\n", answer);
    }

    return error;
}


/* Read all pending errors from the E1562 error queue.  Any errors found
 * are printed to stderr.  The response is the error number from the first
 * in the error queue.
 */
static long check_error(INST e1562, unsigned char echo, const char * msg)
{
    long error;
    long siclError;
    long firstError = 0;
    char result[100];

    for (error = 1; error != 0; )
    {
        siclError = e1562_write(e1562, 0, "syst:err?");
        if (siclError != I_ERR_NOERROR) return siclError;
        siclError = e1562_read(e1562, 0, result, 100);
        if (siclError != I_ERR_NOERROR) return siclError;
        error = atol(result);
        if (error != 0)
        {
            if (firstError == 0)
            {
                if (echo != 0) fprintf(stderr, "%s\n", msg);
                firstError = error;
            }

            if (echo != 0) fprintf(stderr, "\tERROR:  %s", result);
        }
    }

    return firstError;
}


/* Send a command to an E1562 module.  If this command is a query command,
 * read the response from the module.  If the response from the query is
 * not desired, it is permissible to set the answer buffer and/or the answer
 * buffer size to 0.  After the command is sent, all errors are read from
 * the module's error queue.
 */
static long e1562_cmd(INST e1562, unsigned char echo,
                      char * answer, unsigned long answerSize,
                      const char * cmd, ...)
{
    long error;
    va_list args;
    char cmdbuf[CMD_BUFFER_SIZE];

    va_start(args, cmd);
    (void) vsprintf(cmdbuf, cmd, args);
    va_end(args);

    if (!answer || !answerSize)
    {
        /* If the result of a query is not required, it must still be
         * read, but may be discarded.  This just sets the query buffer
         * to a local variable.
         */
        answer = &cmdbuf[strlen(cmdbuf) + 1];
        answerSize = CMD_BUFFER_SIZE - strlen(cmdbuf) - 1;
    }

    if (error = e1562_write(e1562, echo, cmdbuf)) return error;
    if (strchr(cmdbuf, '?'))
    {
        /* This command is a query command, read the response. */
        if (error = e1562_read(e1562, echo, answer, answerSize))
            return error;
    }

    if (cmdbuf[strlen(cmdbuf)-1] == '\n')
        cmdbuf[strlen(cmdbuf)-1] = '\0';

    return check_error(e1562, echo, cmdbuf);
}


/* Find the lowest numbered device on a SCSI bus.  This is done by trying
 * to open each device in succession until the open succeeds.  The device
 * address is returned to the calling function.
 */
static long findFirstDevice(INST e1562, char bus, char* volLetter)
{
    int mode;
    int scsiDeviceNumber = 3;
    int address;
    long err;

#define TM_reserve		0x002
#define TM_ejectOnClose		0x008
#define TM_asynchronousTransfer	0x020
#define TM_preventDisconnect	0x040
#define TM_narrowTransfer   	0x080
#define TM_dontStartUnit	0x100
#define TM_readOnly		0x200

    mode = TM_readOnly | TM_dontStartUnit | TM_narrowTransfer |
	   TM_preventDisconnect | TM_asynchronousTransfer;

    for (address = 0; address < 16; ++address)
    {
        err = e1562_cmd(e1562, 0, 0, 0, "mmem:scsi%d:open %c,%d,0,#h%x",
                        scsiDeviceNumber, bus, address, mode);
        if (err == 0) break;
    }

    if (err != 0) return err;
    if (address >= 16) return -100;

    err = e1562_cmd(e1562, 0, 0, 0, "mmem:scsi%d:clos", scsiDeviceNumber);
    if (err != 0) return err;

    *volLetter = address < 10 ? address + '0' : address + 'a' - 10;

    return 0;
}


/* Find the lowest numbered device on each of the SCSI buses and construct
 * a volume name using these devices.  It is still possible that the
 * default volume name arrived at will not represent a LIF volume, in
 * which case functions trying to use this name will fail.  In order to
 * succeed, the volume name will need to be explicitly specified.
 */
static long findDefaultVolume(INST e1562, char* volume)
{
    char response[CMD_BUFFER_SIZE];
    char* s;
    long err;

    err = e1562_cmd(e1562, 0, response, CMD_BUFFER_SIZE, "*idn?");
    if (err != 0) return err;

    /* Older E1562 modules use ';', newer ones use ',' */
    s = strchr(response, ',');
    if (s == 0) s = strchr(response, ';');
    if (s == 0) return -101; else ++s;
    if (strncmp(s, "E1562", 5)) return -102;
    if (s[5] != 'A' && s[5] != 'B' && s[5] != 'D' && s[5] != 'E')
	return -103;

    volume[0] = 'v';
    if (s[5] == 'A' || s[5] == 'D')
    {
        volume[1] = 'x';
    }
    else
    {
        err = findFirstDevice(e1562, 'A', &volume[1]);
        if (err != 0) return err;
    }

    err = findFirstDevice(e1562, 'B', &volume[2]);
    if (err != 0) return err;

    volume[3] = '\0';

    return 0;
}


/* Convert an ASCII character, as from a volume name, to a SCSI address.
 */
static char charToAddress(char letter)
{
    if (isdigit(letter)) return letter - '0';
    else                 return tolower(letter) - 'a' + 10;
}


/* Convert a volume name into an array of devices needed to construct
 * that volume name.  Each element in the array is a long where the lower
 * 16 bits is the SCSI address, and the upper 16 bits indicates whether
 * the device is on the E1562 A SCSI bus (0), or the E1562 B SCSI bus (1).
 */
static long volumeNameToArray(const char* name, long* array, long* numBlocks,
                              ScsiBusType* busType)
{
    const char* s;
    unsigned int count = 0;
    unsigned int tunitCount = 0;

    *busType = SBT_UNKNOWN;
    if (!name) return -104;

    /* skip white space */
    for (s = name; *s == ' ' || *s == '\t'; ++s) ;

    /* parse the volume name */
    while (*s && *s != ':')
    {
        /* The first letter of every transfer unit in the volume is a 'v'. */
        if (tolower(*s) != 'v') return -105;
        ++s;

        /* This is the A bus */
        if (isxdigit(*s))
        {
            switch (*busType)
            {
            case SBT_UNKNOWN: *busType = SBT_A_ONLY; break;
            case SBT_A_AND_B:
            case SBT_A_ONLY: break;
            case SBT_B_ONLY: return -106;
            default: return -107;
            }

            array[count++] = ((long)eScsiA << 16) | (long)charToAddress(*s);
        }
        else if (tolower(*s) != 'x') return -108;
        ++s;

        /* This is the B bus */
        if (isxdigit(*s))
        {
            switch (*busType)
            {
            case SBT_UNKNOWN: *busType = SBT_B_ONLY; break;
            case SBT_A_AND_B:
            case SBT_B_ONLY: break;
            case SBT_A_ONLY: *busType = SBT_A_AND_B; break;
            default: return -109;
            }

            array[count++] = ((long)eScsiB << 16) | (long)charToAddress(*s);
        }
        else if (tolower(*s) != 'x') return -110;
        ++s;

        /* Now check for a transfer unit size */
        if (isdigit(*s))
        {
            numBlocks[tunitCount++] = atol(s);
            while (isdigit(*s)) ++s;
        }
        else if (tunitCount > 0)
        {
            /* This is not the first transfer unit and a size was not
             * specified -- use the same size as the first transfer unit.
             */
            numBlocks[tunitCount++] = numBlocks[0];
        }
        else if (tunitCount == 0 && (!*s || *s == ':' || tolower(*s) == 'v'))
        {
            /* This is the first transfer unit and a size was not
             * specified -- use the default transfer unit size.
             */
            numBlocks[tunitCount++] = DEFAULT_TUNIT_BLOCKS;
        }
    }

    if (count == 0) return -111;

    /* Now make sure that if a pair of devices is used for one transfer unit,
     * that a pair is used for all transfer units.
     */
    if (*busType == SBT_A_AND_B)
    {
        int i;

        if (count & 1) return -112;

        for (i = 0; i < count; ++i)
        {
            if (!(i & 1) && (array[i] >> 16) == eScsiB)
                return -113;
            else if ((i & 1) && (array[i] >> 16) == eScsiA)
                return -114;
        }
    }

    /* Mark the end of the list of devices. */
    array[count] = -1;
    numBlocks[tunitCount] = -1;

    return 0;
}


/* Given the list of SCSI devices making up a volume and the associated
 * size of each transfer unit, open all devices, transfer units and a
 * session such that the specified volume can be read and written.  This
 * session will be used for the throughput record and playback, as the LIF
 * code uses its own set of devices.  firstScsi specifies the first device
 * number to use when opening devices.  The session will be returned
 * in sessionUsed.
 */
static long open_session(INST e1562, long* devList, long* tunitSize,
                         unsigned short firstScsi, short* sessionUsed)
{
    long err = 0;
    short sess = ((short) firstScsi + 1) / 2;
    short deviceId = firstScsi;
    short devIndex;
    short scsiA, scsiB;

    /* ASSUMPTION:  firstScsi is odd. */

    /* The following loop will open a transfer unit once per loop.  Device
     * ids start at firstScsi and increment for every potential device in
     * the volume.  Transfer unit ids are always equal to the second
     * device id of the loop divided by 2.  The session id is the same
     * as the first transfer unit id.
     */

    /* Clear out the session to be used. */
    err = e1562_cmd(e1562, 0, 0, 0, "mmem:sess%u:del:all", (int)sess);
    if (err != 0) return err;

    /* Loop through the entire device list. */
    for (devIndex = 0; devList[devIndex] >= 0; )
    {
        if ((devList[devIndex] >> 16) == eScsiA)
        {
            scsiA = (short)(devList[devIndex] & 0xf);

            err = e1562_cmd(e1562, 0, 0, 0, "mmem:scsi%d:clos;*cls\n",
                            (int)deviceId);
            if (err != 0) return err;

            err = e1562_cmd(e1562, 0, 0, 0, "mmem:scsi%d:open A,%d,0,#h40\n",
                            (int)deviceId, (int)scsiA);
            if (err != 0) return err;

            err = e1562_cmd(e1562, 0, 0, 0, "mmem:scsi%d:cal:auto off\n",
                            (int)deviceId);
            if (err != 0) return err;

            /* Should a cal be forced now? */

            ++devIndex;
        }
        else
        {
            scsiA = -1;
        }

        ++deviceId;

        if ((devList[devIndex] >> 16) == eScsiB)
        {
            scsiB = (short)(devList[devIndex] & 0xf);

            err = e1562_cmd(e1562, 0, 0, 0, "mmem:scsi%d:clos;*cls\n",
                            (int)deviceId);
            if (err != 0) return err;

            err = e1562_cmd(e1562, 0, 0, 0, "mmem:scsi%d:open B,%d,0,#h40\n",
                            (int)deviceId, (int)scsiB);
            if (err != 0) return err;

            err = e1562_cmd(e1562, 0, 0, 0, "mmem:scsi%d:cal:auto off\n",
                            (int)deviceId);
            if (err != 0) return err;

            /* Should a cal be forced now? */

            ++devIndex;
        }
        else
        {
            scsiB = -1;
        }


        /* Open a transfer unit */

        err = e1562_cmd(e1562, 0, 0, 0, "mmem:tun%d:clos;*cls\n",
                        (int)(deviceId / 2));
        if (err != 0) return err;

        if (scsiA >= 0 && scsiB >= 0)
        {
            err = e1562_cmd(e1562, 0, 0, 0, "mmem:tun%d:open %d,0,%d,0\n",
                            (int)(deviceId / 2),
                            (int)(deviceId - 1), (int)deviceId);
            if (err != 0) return err;
        }
        else if (scsiA >= 0)
        {
            err = e1562_cmd(e1562, 0, 0, 0, "mmem:tun%d:open %d,0\n",
                            (int)(deviceId / 2),
                            (int)(deviceId - 1));
            if (err != 0) return err;
        }
        else
        {
            err = e1562_cmd(e1562, 0, 0, 0, "mmem:tun%d:open %d,0\n",
                            (int)(deviceId / 2),
                            (int)deviceId);
            if (err != 0) return err;
        }

        /* Add the transfer unit to the session. */
        err = e1562_cmd(e1562, 0, 0, 0, "mmem:sess%u:add %d,%ld\n",
                        (int)sess,
                        (int)(deviceId / 2),
                        tunitSize[(deviceId / 2) - sess]);
        if (err != 0) return err;

        ++deviceId;
    }

    *sessionUsed = sess;

    return 0;
}


/* Open all devices needed to address a volume given the name of the volume.
 */
static long openSession(INST e1562, const char* volume, short* sessionNumber)
{
    long devList[32];
    long tunitSize[16];
    ScsiBusType bus;
    long err;

    err = volumeNameToArray(volume, devList, tunitSize, &bus);
    if (err != 0) return err;

    err = open_session(e1562, devList, tunitSize, DEFAULT_SESSION_NUMBER * 2,
                       sessionNumber);
    if (err != 0) return err;

    return 0;
}


/* Make the initial contact with the E1562.  This function performs an
 * iopen, and initializes the E1562 LIF library.  If the LIF library is
 * already in use, this may cause errors.
 */
long e1562tput_open(const char* interface, short e1562LA,
                    E1562Handle* handle)
{
    char buffer[100];
    INST e1562;
    long err;
    ThroughputInformation* h;
    e1562_errors e1562Err;
    short i;

    *handle = 0;

    for (i = 0; i < MAX_OPEN_FILES; ++i)
        if (openFiles[i] == 0) break;
    if (i >= MAX_OPEN_FILES) return -130;

    (void) sprintf(buffer, "%s,%d", interface, e1562LA);
    e1562 = iopen(buffer);
    if (e1562 == 0) return -115;

    iclear(e1562);

    /* Make sure that the E1562 is in the default state.  This is done by
     * resetting the module and waiting for it to finish.
     */
    err = e1562_cmd(e1562, 0, 0, 0, "*rst;*cls;*opc?");
    if (err != 0) return err;

    e1562Err = e1562_initializeLibrary();
    if (e1562Err != e1562Err_noError) return (long)e1562Err;

    e1562Err = e1562_mapModule(E1562_ID, interface, e1562LA);
    if (e1562Err != e1562Err_noError) return (long)e1562Err;

    h = (ThroughputInformation*)malloc(sizeof(ThroughputInformation));
    if (h == 0) return -116;

    h->id = e1562;
    h->lifid = E1562_ID;
    h->file = 0;
    h->totalBytes = 0.0;
    h->scsiBlock = 0;
    h->scsiBlocksize = 0;
    h->session = 0;
    h->splitVolume = 0;
    h->sequenceIsThroughput = 1;
    h->logicalAddress = e1562LA;
    h->interface = (char*)malloc(strlen(interface) + 1);
    if (h->interface == 0) return -200;
    (void) strcpy(h->interface, interface);
    h->debugLevel = 0;

    openFiles[i] = h;
    *handle = i;

    privateRecordFinished = 1;
    privateRecordBytes = 0.0;

    h->sram = (short *) (void *) imap(e1562, I_MAP_EXTEND, 0, 4, 0);
    if (h->sram == 0)
    {
        if (h->debugLevel > 0) fprintf(stderr, "Can't map E1562 shared RAM\n");
        return -121;
    }

    return 0;
}


/* Open an E1562 LIF file in preparation for a throughput record.  The
 * total file size must be known at open time, as a LIF file cannot be
 * extended in size after it has been created.  This means that the value
 * of totalBytes must include the size of the data to be recorded, the
 * size of any header information, and the size of any information
 * following the recorded data.  The value passed to totalBytes will
 * be padded with one SCSI blocksize, since the recorded data must start
 * at a SCSI block boundary.  This will mean that any directory listing
 * may show a different size for the file than what was passed to this
 * function.  The file pointer is returned to the calling function with
 * the file open.  It is recommended that some information about the
 * recorded data be written to the file when this function returns.  Also,
 * since this routine may be called with just a filename allowing the
 * volume to be defaulted, the full filename is returned to the calling
 * function.  It is possible to default the volume name only if a single
 * disk is used as the LIF volume, or if a pair of disks, one on each
 * SCSI bus of the E1562, is used as the LIF volume.  Also, the SCSI devices
 * must be the lowest addressed devices on the SCSI bus.  If the volume name
 * is specified, any number of SCSI devices may be used to make up the
 * file system.  This function expects that a file system already exists
 * on the devices -- this can be done using the e1562in command from the
 * E1562 LIF utilites.  Since the filename may be defaulted, the full
 * filename is returned in case the file needs to be opened again later.
 */
long e1562tput_openRecordFile(E1562Handle e1562Handle,
                              char* filename,
                              double totalBytes)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    char volume[e1562_FILENAME_MAX];
    char name[e1562_FILENAME_MAX];
    short session;
    char* s;
    long err;

    if (e1562 == 0) return -131;

    (void) strcpy(volume, filename);
    s = strchr(volume, ':');
    if (s == 0)
    {
        err = findDefaultVolume(e1562->id, volume);
        if (err != 0) return err;
        (void) sprintf(name, "%s:", volume);
        s = filename;
    }
    else
    {
        name[0] = '\0';
        *s = '\0';
        ++s;
    }

    (void) sprintf(name, "%s:%s", volume, s);
    (void) strcpy(filename, name);

    err = openSession(e1562->id, volume, &session);
    if (err != 0) return err;
    e1562->session = session;
    e1562->splitVolume = (tolower(volume[1]) != 'x' &&
                          tolower(volume[2]) != 'x');

    /* Add enough bytes to act as a pad factor if the beginning of the
     * data in the throughput record needs to be adjusted to the beginning
     * of the next SCSI block.  One disk has a blocksize of 512 bytes
     * (usually), a volume split across both SCSI buses has an effective
     * blocksize of 1024.  If the blocksize of the SCSI devices is not 512,
     * this padding factor should be changed accordingly.
     */
    totalBytes += 1024.0;
    e1562->totalBytes = totalBytes;

    /* Remove any existing file of the same name, since it might be
       too small and if so the thruput will fail. */
    (void) e1562_remove(e1562->lifid, filename);

    (void) sprintf(name, "%s,%.17g", filename, totalBytes);
/*    printf("Opening file %s\n", name); */
    e1562->file = e1562_fopen(e1562->lifid, name, "w");
    if (e1562->file == 0)
    {
        if (e1562->debugLevel > 10)
            fprintf(stderr, "e1562_fopen(%s, \"wb\") failed with errno = %d\n",
                    name, (int)e1562_errno);
        return -117;
    }

#if 0
    {
    unsigned long scsiBlockNumber;
    unsigned long scsiBlocksize;
    unsigned long scsiBytesInBlock;
    scsiBlockNumber = e1562_block(e1562->file, &scsiBlocksize,
                                  &scsiBytesInBlock);
    if (e1562->debugLevel > 10)
        printf("e1562_block returned:  block=%lu, padBytes=%lu, "
               "blocksize=%lu\n",
               scsiBlockNumber, scsiBytesInBlock, scsiBlocksize);
    }
#endif

    privateRecordFinished = 1;
    privateRecordBytes = 0.0;

    return 0;
}


long e1562tput_openFileForUpdate(E1562Handle e1562Handle,
                                 char* fullFileName)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    e1562_errors e1562Err;

    if (e1562 == 0) return -131;

    /* This close, then open sequence is done to bypass a bug in the E1562
     * LIF library code which sometimes will prevent an existing file from
     * being opened.
     */
    ionsrq(e1562->id, 0);

    e1562Err = e1562_closeLibrary();    /* close the open library */

    iclear(e1562->id);

    e1562Err = e1562_initializeLibrary();
    if (e1562Err != e1562Err_noError)
    {
	(void) fprintf(stderr, "Error in init library\n");
	return (long) e1562Err;
    }

    e1562Err = e1562_mapModule(e1562->lifid, e1562->interface,
                               e1562->logicalAddress);
    if (e1562Err != e1562Err_noError)
    {
	(void) fprintf(stderr, "Error in map module\n");
	return (long) e1562Err;
    }

    e1562->file = e1562_fopen(e1562->lifid, fullFileName, "r+");
    if (e1562->file == 0) return -118;

#if 0
    {
    unsigned long scsiBlockNumber;
    unsigned long scsiBlocksize;
    unsigned long scsiBytesInBlock;
    scsiBlockNumber = e1562_block(e1562->file, &scsiBlocksize,
                                  &scsiBytesInBlock);
    if (e1562->debugLevel > 10)
        printf("e1562_block returned:  block=%lu, padBytes=%lu, "
               "blocksize=%lu\n",
               scsiBlockNumber, scsiBytesInBlock, scsiBlocksize);
    }
#endif

    return 0;
}


/* Open an E1562 LIF file in preparation for a throughput post-process.  The
 * open file's file pointer is returned so the application may read any
 * header information contained in the file.  Also, as the volume name may
 * be defaulted, the full filename including the volume is returned.
 */
long e1562tput_openPlaybackFile(E1562Handle e1562Handle,
                                char* filename)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    char volume[e1562_FILENAME_MAX];
    char name[e1562_FILENAME_MAX];
    short session;
    char* s;
    long err;

    if (e1562 == 0) return -131;

    (void) strcpy(volume, filename);
    s = strchr(volume, ':');
    if (s == 0)
    {
        err = findDefaultVolume(e1562->id, volume);
        if (err != 0) return err;
        (void) sprintf(name, "%s:", volume);
        s = (char*)filename;
    }
    else
    {
        name[0] = '\0';
        *s = '\0';
        ++s;
    }

    (void) sprintf(name, "%s:%s", volume, s);
    (void) strcpy(filename, name);

    err = openSession(e1562->id, volume, &session);
    if (err != 0) return err;
    e1562->session = session;
    e1562->splitVolume = (tolower(volume[1]) != 'x' &&
                          tolower(volume[2]) != 'x');

/*    printf("Opening file %s\n", fullFileName); */
    e1562->file = e1562_fopen(e1562->lifid, filename, "r");
    if (e1562->file == 0) return -118;

    privateRecordFinished = 1;
    privateRecordBytes = 0.0;

    return 0;
}


/* Initialize the E1562 sequence to perform a local bus throughput with
 * the specified number of inputs and blocksize.  A constant blocksize for
 * all channels is assumed.  An E1562 session is setup to write at the
 * next SCSI block boundary in the file opened in e1562tput_openRecordFile().
 * The open file is then closed, as the session is used for the throughput
 * instead of the file.  Any header information to be written to the file
 * should be written before calling this function as the current position
 * of the file pointer is used to determine where to start writing the data.
 * The number of bytes needed for padding is returned so that the offset in
 * the file to the recorded data can be determined.  If the pad is non-zero,
 * the header should be written again at the end of the record.
 */
long e1562tput_setupRecord(E1562Handle e1562Handle,
                           unsigned long bytesPerInputBlock,
                           short numberInputs,
                           unsigned long* retPadBytes)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    e1562_errors e1562Err;
    long err;
    e1562_fpos_t offset;
    int sequenceNumber = DEFAULT_SEQUENCE_NUMBER;
    unsigned long scsiBlockNumber;
    unsigned long scsiBlocksize;
    unsigned long scsiBytesInBlock;

    if (e1562 == 0) return -131;

    /* Setup the E1562 sequence */
    err = e1562_cmd(e1562->id, 0, 0, 0, "seq%d:del:all;*cls",
                    sequenceNumber);
    if (err != 0) return err;

    err = e1562_cmd(e1562->id, 0, 0, 0, "seq%d:add #h%.4x,%lu,%lu,#h%.8x",
                    sequenceNumber, Seq_lbusConsume, numberInputs, 0,
                    bytesPerInputBlock | (3 << 24UL));
    if (err != 0) return err;

    e1562->sequenceIsThroughput = 1;


    /* Get current position in the file */
    scsiBlockNumber = e1562_block(e1562->file, &scsiBlocksize,
                                  &scsiBytesInBlock);
    if (scsiBlockNumber == 0xffffffffUL) return -119;

    /* Adjust offset to nearest SCSI block boundry */
    if (scsiBytesInBlock > 0)
    {
        ++scsiBlockNumber;
        *retPadBytes = scsiBlocksize - scsiBytesInBlock;
    }
    else
    {
        *retPadBytes = 0;
    }

    if (e1562->splitVolume != 0) scsiBlockNumber *= 2;

    /* Make sure the logical size specifies the whole file. */
#if defined(USE_INTEGRAL_OFFSET)
    offset = (e1562_fpos_t)e1562->totalBytes;
#else
    offset.positionHigh = (unsigned long)floor(e1562->totalBytes /
                                               ((double)ULONG_MAX + 1.0));
    offset.positionLow = (unsigned long)fmod(e1562->totalBytes,
                                             ((double)ULONG_MAX + 1.0));
#endif
    e1562Err = e1562_setEOF(e1562->file, &offset);
    if (e1562Err != e1562Err_noError) return (long)e1562Err;

    e1562->scsiBlock = scsiBlockNumber;
    e1562->scsiBlocksize = scsiBlocksize;

    if (e1562->debugLevel > 10)
        printf("e1562_block returned:  block=%lu, padBytes=%lu, "
               "blocksize=%lu\n",
               scsiBlockNumber, *retPadBytes, scsiBlocksize);

    e1562Err = e1562_fclose(e1562->file);
    if (e1562Err != e1562Err_noError) return (long)e1562Err;

    privateRecordFinished = 1;
    privateRecordBytes = 0.0;

    return 0;
}


/* Initialize the E1562 sequence to perform a playback using an E1562
 * sequence to read bytes into the shared RAM on the E1562.  Synchronization
 * between the host program and the E1562 is done through the use of the
 * first 2 bytes of E1562 shared RAM as a flag.  By using the routine
 * e1562tput_readPlaybackScan(), all synchronization is encapsulated in
 * these example routines.  Be aware that the maximum value for bytesPerScan
 * is 262142, although there are no checks for exceeding this value.  A
 * session is setup to be used with the E1562 sequence.  The open playback
 * file is closed.  The function e1562tput_readPlaybackScan is used to
 * perform the required synchronization with the E1562 after
 * e1562tput_startPlayback has been called.
 */
long e1562tput_setupPlayback(E1562Handle e1562Handle,
                             unsigned long dataOffset,
                             unsigned long bytesPerScan)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    e1562_errors e1562Err;
    long err;
    int sequenceNumber = DEFAULT_SEQUENCE_NUMBER;
    unsigned long scsiBlockNumber;
    unsigned long scsiBlocksize;
    unsigned long scsiBytesInBlock;
    e1562_fpos_t offset;

    if (e1562 == 0) return -131;

    if (bytesPerScan > 262142)
    {
	(void) printf("Block size too big for E1562 playback\n");
	return -1;
    }

#if 0
    scsiBlockNumber = e1562_block(e1562->file, &scsiBlocksize,
                                  &scsiBytesInBlock);
    if (e1562->debugLevel > 10)
        printf("e1562_block returned:  block=%lu, padBytes=%lu, "
               "blocksize=%lu\n",
               scsiBlockNumber, scsiBytesInBlock, scsiBlocksize);
#endif

    /* Setup the E1562 sequence.  First clear the sequence. */
    err = e1562_cmd(e1562->id, 0, 0, 0, "seq%d:del:all;*cls",
                    sequenceNumber);
    if (err != 0) return err;

    /* Add a sequence operation to wait for the 16-bit value at offset 0 in
     * the E1562 shared RAM to exceed 1.
     */
    err = e1562_cmd(e1562->id, 0, 0, 0, "seq%d:add #h%.4x,%lu,%lu,#h%.8x",
                    sequenceNumber, Seq_waitCountSharedRam16, 1, 0, 100);
    if (err != 0) return err;

    /* Add a sequence operation to write the next N bytes from the data
     * stream into E1562 shared RAM.  If it is desired to use scan sizes
     * larger than 262142 and some other RAM is available in VXI A24
     * space, it might be better to use that RAM instead of teh E1562 RAM.
     */
    err = e1562_cmd(e1562->id, 0, 0, 0, "seq%d:add #h%.4x,%lu,%lu,#h%.8x",
                    sequenceNumber, Seq_writeSharedRam, bytesPerScan, 2, 0);
    if (err != 0) return err;

    /* Add a sequence operation to clear the 16-bit value at offset 0 in
     * the E1562 shared RAM.
     */
    err = e1562_cmd(e1562->id, 0, 0, 0, "seq%d:add #h%.4x,%lu,%lu,#h%.8x",
                    sequenceNumber, Seq_controlRegSharedRam16, 0, 0, 0);
    if (err != 0) return err;

    e1562->sequenceIsThroughput = 0;


    /* printf("dataOffset=%lu\n", dataOffset); */

    /* Seek to the beginning of the data in the file */
#if defined(USE_INTEGRAL_OFFSET)
    offset = (e1562_fpos_t)dataOffset;
#else
    offset.positionHigh = (unsigned long)floor((double)dataOffset /
                                               ((double)ULONG_MAX + 1.0));
    offset.positionLow = (unsigned long)fmod((double)dataOffset,
                                             ((double)ULONG_MAX + 1.0));

    /* printf("high=%lu  low=%lu\n",offset.positionHigh, offset.positionLow);*/
#endif
    e1562Err = e1562_fsetpos(e1562->file, &offset);
    if (e1562Err != e1562Err_noError) return (long)e1562Err;

    /* Get current position in the file */
    scsiBlockNumber = e1562_block(e1562->file, &scsiBlocksize,
                                  &scsiBytesInBlock);
    if (scsiBlockNumber == 0xffffffffUL) return -120;

    /* Adjust offset to nearest SCSI block boundry */
    if (scsiBytesInBlock > 0)
    {
        printf("ERROR:  e1562tput_setupPlayback adjusting SCSI block\n");
        ++scsiBlockNumber;
    }

    if (e1562->splitVolume != 0) scsiBlockNumber *= 2;

    e1562->scsiBlock = scsiBlockNumber;
    e1562->scsiBlocksize = scsiBlocksize;

    if (e1562->debugLevel > 10)
        printf("e1562_block returned:  block=%lu, padBytes=%lu, "
               "blocksize=%lu\n",
               scsiBlockNumber, scsiBytesInBlock, scsiBlocksize);

    e1562Err = e1562_fclose(e1562->file);
    if (e1562Err != e1562Err_noError) return (long)e1562Err;

    return 0;
}


static void SRQhandler(INST id)
{
    char response[CMD_BUFFER_SIZE];
    unsigned char status;
    long err;

    /* Read the status byte of the E1562 to determine why it interrupted. */
    err = ireadstb(id, &status);
    if (err != I_ERR_NOERROR)
    {
        fprintf(stderr, "Error reading E1562 status\n");
        return;
    }

    if ((status & 0x80) != 0)
    {
        /* The SRQ was caused by the sequence completing. */
        privateRecordFinished = 1;

        err = e1562_cmd(id, 0, response, CMD_BUFFER_SIZE,
                        "seq%d:tran?", DEFAULT_SEQUENCE_NUMBER);
        if (err != 0)
        {
            fprintf(stderr, "Error reading E1562 throughput size\n");
            (void) strcpy(response, "0.0");
        }

        privateRecordBytes = atof(response);

        if (privateCallbackFunc != 0)
        {
            privateCallbackFunc(privateHandle, privateCallbackArg,
                                privateRecordBytes);
        }
    }
}


/* Start the E1562 record sequence running.  If a callback function is
 * specified, an SRQ is setup such that a function call will be made
 * to the specified routine when the throughput record is finished.  The
 * specified argument and the number of bytes recorded will be passed
 * to the callback routine.  If no callback is desired, callbackFunc may
 * be specified as 0.
 */
long e1562tput_startRecord(E1562Handle e1562Handle,
                           double lengthInBytes,
                           void (*callbackFunc)(E1562Handle, volatile void*,
                                                double),
                           volatile void* callbackArg)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    char response[CMD_BUFFER_SIZE];
    long err;
    void (*oldSRQhandler)();

    if (e1562 == 0) return -131;

    /* Tell the E1562 to seek to the point where the Y data begins. */
    err = e1562_cmd(e1562->id, 0, 0, 0, "mmem:sess%d:seek %lu",
                    e1562->session, e1562->scsiBlock);
    if (err != 0) return err;

    /* This delay is required due to a race condition in the E1562 firmware
     * which causes the seq:beg command below to execute before the seek has
     * had time to finish.
     */
    {
	volatile unsigned long j;
	for (j = 0;  j < 1000000;  ++j) ;
    }

    /* Setup the E1562 status registers to cause an SRQ when the sequence
     * has finished.
     */
    err = e1562_cmd(e1562->id, 0, 0, 0,
                    "*cls;:stat:oper:ptr 0;ntr 16;enab 16;*sre 128");
    if (err != 0) return err;

    privateCallbackFunc = callbackFunc;
    privateCallbackArg = callbackArg;
    privateHandle = e1562Handle;
    privateRecordFinished = 0;
    privateRecordBytes = 0.0;

    /* Setup SRQ processing */
    err = igetonsrq(e1562->id, &oldSRQhandler);
    if (err != I_ERR_NOERROR) return err;

    err = ionsrq(e1562->id, SRQhandler);
    if (err != I_ERR_NOERROR) return -1;

    e1562->sequenceIsThroughput = 1;

    err = e1562_cmd(e1562->id, 0, 0, 0, "seq%d:beg thr,%.17g,%ld",
                    DEFAULT_SEQUENCE_NUMBER, lengthInBytes,
                    e1562->session);
    if (err != 0) return err;

    (void) strcpy(response, "0");
    while (atol(response) == 0)
    {
        err = e1562_cmd(e1562->id, 0, response, CMD_BUFFER_SIZE,
                        "stat:oper:cond?");
        if (err != 0) return err;
    }

    return 0;
}


long e1562tput_recordFinished(E1562Handle e1562Handle, long* flag)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];

    if (e1562 == 0) return -131;

    *flag = (long)privateRecordFinished;

    return 0;
}


long e1562tput_recordBytes(E1562Handle e1562Handle, double* bytes)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];

    if (e1562 == 0) return -131;

    *bytes = privateRecordBytes;

    return 0;
}


/* Perform the required synchronization with the E1562 to get bytes
 * transferred into its shared RAM and copy the bytes into the host.  The
 * value byteCount should be less than or equal to 262142 and should be the
 * same as the value bytesPerScan passed to e1562tput_setupPlayback.
 */
long e1562tput_readPlaybackScan(E1562Handle e1562Handle,
                                unsigned long byteCount, void* buf)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    long error;
    unsigned char status;

    if (e1562 == 0) return -131;

    /* Wait for data to be ready */
    while (iwpeek(e1562->sram) != 0)
    {
        error = ireadstb(e1562->id, &status);
        if (error != I_ERR_NOERROR)
        {
            fprintf(stderr, "Error reading E1562 status\n");
            return error;
        }

        if ((status & 0x80) != 0)
        {
            /* The E1562 sequence has finished, so the flag is never going
             * to be set.
             */
            break;
        }
    }

    /* Read the data into host memory. */
    error = iwblockcopy(e1562->id, (unsigned short *)e1562->sram + 1, 
                        (unsigned short *)buf,
                        (unsigned long)(byteCount / 2), 0);

    if (error != I_ERR_NOERROR) return error;

    /* Tell the E1562 to load more data */
    iwpoke(e1562->sram, 2);

    return 0;
}


/* Start the E1562 playback sequence running.  If a callback function is
 * specified, an SRQ is setup such that a function call will be made
 * to the specified routine when the throughput playback is finished.  The
 * specified argument and the number of bytes read will be passed
 * to the callback routine.  The argument firstScanOffset is the scan
 * number of the first scan when the throughput file was treated as a
 * circular buffer -- it is zero otherwise.  If a callback function is
 * not needed, pass 0 as the argument callbackFunc.
 */
long e1562tput_startPlayback(E1562Handle e1562Handle,
                             double lengthInBytes,
                             unsigned long firstScanOffset,
                             void (*callbackFunc)(E1562Handle,
                                                  volatile void*, double),
                             volatile void* callbackArg)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    char response[CMD_BUFFER_SIZE];
    long err;
    unsigned long scan;
    unsigned char status;

    if (e1562 == 0) return -131;

    /* Tell the E1562 to seek to the point where the Y data begins. */
    err = e1562_cmd(e1562->id, 0, 0, 0, "mmem:sess%d:seek %lu",
                    e1562->session, e1562->scsiBlock);
    if (err != 0) return err;

    /* Setup the E1562 status registers to cause an SRQ when the sequence
     * has finished.
     */
    err = e1562_cmd(e1562->id, 0, 0, 0,
                    "*cls;:stat:oper:ptr 0;ntr 16;enab 16;*sre 128");
    if (err != 0) return err;

    privateCallbackFunc = callbackFunc;
    privateCallbackArg = callbackArg;
    privateHandle = e1562Handle;

    if (callbackFunc != 0)
    {
        void (*oldSRQhandler)();

        /* Setup SRQ processing */
        err = igetonsrq(e1562->id, &oldSRQhandler);
        if (err != I_ERR_NOERROR) return err;

        err = ionsrq(e1562->id, SRQhandler);
        if (err != I_ERR_NOERROR) return err;
    }

    /* Tell the E1562 to load the first scan */
    iwpoke(e1562->sram, 2);
    e1562->sequenceIsThroughput = 0;
    e1562->totalBytes = lengthInBytes;


    /* Tell the E1562 to begin the playback sequence. */
    err = e1562_cmd(e1562->id, 0, 0, 0, "seq%d:beg vpl,%.17g,%ld",
                    DEFAULT_SEQUENCE_NUMBER, lengthInBytes,
                    e1562->session);
    if (err != 0) return err;

    /* Wait for the sequence to "really" be started!
     */
    (void) strcpy(response, "0");
    while (atol(response) == 0)
    {
        err = e1562_cmd(e1562->id, 0, response, CMD_BUFFER_SIZE,
                        "stat:oper:cond?");
        if (err != 0) return err;
    }


    /* If the firstScanOffset is non-zero, the data was recorded using
     * a circular buffer.  Data needs to be discarded until we have reached
     * the scan in which the data begins.
     */
    for (scan = 0; scan < firstScanOffset; ++scan)
    {
        /* Wait for data to be ready */
        while (iwpeek(e1562->sram) != 0)
        {
            err = ireadstb(e1562->id, &status);
            if (err != I_ERR_NOERROR)
            {
                fprintf(stderr, "Error reading E1562 status\n");
                return err;
            }

            if ((status & 0x80) != 0)
            {
                fprintf(stderr, "Bad E1562 status: 0x%.2x\n", status);
                return -121;
            }
        }

        /* Tell the E1562 to load more data */
        iwpoke(e1562->sram, 2);
    }

    return 0;
}


/* Seek to an absolute location in a file.
 */
long e1562tput_seek(E1562Handle e1562Handle, double dataOffset)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    e1562_fpos_t offset;
    e1562_errors e1562Err;

    if (e1562 == 0) return -131;

    /* Seek to the beginning of the data in the file */
#if defined(USE_INTEGRAL_OFFSET)
    offset = (e1562_fpos_t)dataOffset;
#else
    offset.positionHigh = (unsigned long)floor(dataOffset /
                                               ((double)ULONG_MAX + 1.0));
    offset.positionLow = (unsigned long)fmod(dataOffset,
                                             ((double)ULONG_MAX + 1.0));

    /* printf("high=%lu  low=%lu\n",offset.positionHigh, offset.positionLow);*/
#endif
    e1562Err = e1562_fsetpos(e1562->file, &offset);
    if (e1562Err != e1562Err_noError) return (long)e1562Err;

    return 0;
}


long e1562tput_read(void* buf, size_t elsize, size_t count,
                    E1562Handle e1562Handle)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];

    if (e1562 == 0) return -131;

    return e1562_fread(buf, elsize, count, e1562->file);
}


long e1562tput_write(void* buf, size_t elsize, size_t count,
                     E1562Handle e1562Handle)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];

    if (e1562 == 0) return -131;

    return e1562_fwrite(buf, elsize, count, e1562->file);
}


/* Terminate access to the E1562.  The LIF library is closed and the E1562
 * SICL id is issued an iclose().  Use of the E1562 handle after the call
 * to this routine is not allowed and may cause unexpected failures.
 */
long e1562tput_close(E1562Handle e1562Handle)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    short i;

    if (e1562 == 0) return -131;

    for (i = 0; i < MAX_OPEN_FILES; ++i)
    {
        if (openFiles[i] != 0 && openFiles[i]->file != 0)
            e1562_fclose(openFiles[i]->file);
    }

    (void) ionsrq(e1562->id, 0);
    (void) e1562_closeLibrary();    /* close the open session */
    iclose(e1562->id);
    free(e1562);

    return 0;
}


/* Allow the application to put the E1562 local bus into reset, or take
 * it out of reset.  This is needed to provide for the safe reset of all
 * devices on the local bus.  For example, a safe reset consists of first
 * placing all adjacent local bus devices into reset, then from left to
 * right in the VXI cardcage, take each device's local bus out of reset.
 */
long e1562tput_resetE1562lbus(E1562Handle e1562Handle, short putIntoReset)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    long err;

    if (e1562 == 0) return -131;

    err = e1562_cmd(e1562->id, 0, 0, 0, "vins:lbus %s",
                    putIntoReset ? "reset" : "normal");
    if (err != 0) return err;

    return 0;
}


/* Halt an E1562 sequence.  This can be used to abort a throughput record
 * before it would normally complete, or in the case of a circular buffer,
 * to stop the throughput record.  This function may also be used to stop
 * a playback sequence early and place the E1562 in an idle state.  For both
 * a record or playback sequence, if a callback function was specified, it
 * is called with the number of bytes which were actually throughout.
 */
long e1562tput_abortThroughput(E1562Handle e1562Handle)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];

    if (e1562 == 0) return -131;

    return e1562_write(e1562->id, 0, "syst:abor");
}


/* Provide for the ability to implement a circular buffer in a throughput
 * file.  For instance, with the use of a callback routine when the throughput
 * is finished, this function will allow the sequence to be restarted at the
 * beginning of the recorded data.  Some application logic is required to
 * determine when to stop restarting the sequence, or when to abort it.
 */
long e1562tput_continueThroughput(E1562Handle e1562Handle)
{
    ThroughputInformation* e1562 = openFiles[e1562Handle];
    unsigned char status;
    long err;

    if (e1562 == 0) return -131;

    err = ireadstb(e1562->id, &status);
    if (err != I_ERR_NOERROR)
    {
        fprintf(stderr, "Error reading E1562 status\n");
        return err;
    }

    if ((status & 0x80) != 0)
    {
        if (e1562->sequenceIsThroughput == 0)
        {
            char response[CMD_BUFFER_SIZE];
            /* This is a playback sequence.  The E1562 SEQ:CONT command
             * only works for throughput sequences, so it is necessary
             * to perform the seek and start manually.
             */

            /* Tell the E1562 to seek to the point where the Y data begins */
            err = e1562_cmd(e1562->id, 0, 0, 0, "mmem:sess%d:seek %lu",
                            e1562->session, e1562->scsiBlock);
            if (err != 0) return err;

            /* Clear the data ready flag, because the E1562 sequence
             * terminated before it could perform the last flag clear.
             */
            iwpoke(e1562->sram, 0);

            /* Tell the E1562 to begin the playback sequence. */
            err = e1562_cmd(e1562->id, 0, 0, 0,
                            "*cls;:seq%d:beg vpl,%.17g,%ld",
                            DEFAULT_SEQUENCE_NUMBER, e1562->totalBytes,
                            e1562->session);
            if (err != 0) return err;

            /* Wait for the sequence to "really" be started!
             */
            (void) strcpy(response, "0");
            while (atol(response) == 0)
            {
                err = e1562_cmd(e1562->id, 0, response, CMD_BUFFER_SIZE,
                                "stat:oper:cond?");
                if (err != 0) return err;
            }
        }
        else
        {
            /* Tell the E1562 to continue the sequence by seeking back to the
             * the beginning of the session and restarting the sequence.
             * Make sure to clear the status to allow another SRQ to be
             * generated.
             */
            err = e1562_write(e1562->id, 0, "*cls;seq%d:cont %lu",
                              DEFAULT_SEQUENCE_NUMBER, e1562->scsiBlock);
            if (err != 0)
            {
                fprintf(stderr, "Error sending seq:cont  %ld\n", err);
                return err;
            }
        }
    }

    return 0;
}
